<?php
/**
 * Parent/Guardian Model
 * Handles all parent-related database operations
 */
require_once __DIR__ . '/../../core/Database.php';

class ParentModel {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Parents
    public function getAll() {
        $stmt = $this->db->query("SELECT * FROM parents_guardians ORDER BY created_at DESC");
        return $stmt->fetchAll();
    }

    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM parents_guardians WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function create($data) {
        $stmt = $this->db->prepare("INSERT INTO parents_guardians (guardian_type, first_name, last_name, id_number, phone_primary, phone_secondary, email, occupation, employer, address, city, country, postal_code, emergency_contact_name, emergency_contact_phone, relationship_to_student, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['guardian_type'], $data['first_name'], $data['last_name'], $data['id_number'], $data['phone_primary'], $data['phone_secondary'] ?? null, $data['email'] ?? null, $data['occupation'] ?? null, $data['employer'] ?? null, $data['address'] ?? null, $data['city'] ?? null, $data['country'] ?? 'Kenya', $data['postal_code'] ?? null, $data['emergency_contact_name'] ?? null, $data['emergency_contact_phone'] ?? null, $data['relationship_to_student'] ?? null, $data['status'] ?? 1]);
    }

    public function update($id, $data) {
        $stmt = $this->db->prepare("UPDATE parents_guardians SET guardian_type=?, first_name=?, last_name=?, id_number=?, phone_primary=?, phone_secondary=?, email=?, occupation=?, employer=?, address=?, city=?, country=?, postal_code=?, emergency_contact_name=?, emergency_contact_phone=?, relationship_to_student=?, status=? WHERE id=?");
        return $stmt->execute([$data['guardian_type'], $data['first_name'], $data['last_name'], $data['id_number'], $data['phone_primary'], $data['phone_secondary'] ?? null, $data['email'] ?? null, $data['occupation'] ?? null, $data['employer'] ?? null, $data['address'] ?? null, $data['city'] ?? null, $data['country'] ?? 'Kenya', $data['postal_code'] ?? null, $data['emergency_contact_name'] ?? null, $data['emergency_contact_phone'] ?? null, $data['relationship_to_student'] ?? null, $data['status'] ?? 1, $id]);
    }

    public function delete($id) {
        $stmt = $this->db->prepare("DELETE FROM parents_guardians WHERE id = ?");
        return $stmt->execute([$id]);
    }

    public function search($keyword) {
        $stmt = $this->db->prepare("SELECT * FROM parents_guardians WHERE first_name LIKE ? OR last_name LIKE ? OR phone_primary LIKE ? OR email LIKE ? OR id_number LIKE ?");
        $keyword = "%$keyword%";
        $stmt->execute([$keyword, $keyword, $keyword, $keyword, $keyword]);
        return $stmt->fetchAll();
    }

    // Student-Parent Relationship
    public function assignParent($student_id, $parent_id, $is_primary = 0) {
        $stmt = $this->db->prepare("INSERT INTO student_parents (student_id, parent_id, is_primary) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE is_primary=?");
        return $stmt->execute([$student_id, $parent_id, $is_primary, $is_primary]);
    }

    public function getParentsByStudent($student_id) {
        $stmt = $this->db->prepare("SELECT pg.*, sp.is_primary FROM parents_guardians pg 
                                    JOIN student_parents sp ON pg.id = sp.parent_id 
                                    WHERE sp.student_id = ?");
        $stmt->execute([$student_id]);
        return $stmt->fetchAll();
    }

    public function getStudentsByParent($parent_id) {
        $stmt = $this->db->prepare("SELECT s.*, sp.is_primary FROM students s 
                                    JOIN student_parents sp ON s.id = sp.student_id 
                                    WHERE sp.parent_id = ?");
        $stmt->execute([$parent_id]);
        return $stmt->fetchAll();
    }

    public function removeParent($student_id, $parent_id) {
        $stmt = $this->db->prepare("DELETE FROM student_parents WHERE student_id = ? AND parent_id = ?");
        return $stmt->execute([$student_id, $parent_id]);
    }

    public function getPrimaryParent($student_id) {
        $stmt = $this->db->prepare("SELECT pg.* FROM parents_guardians pg 
                                    JOIN student_parents sp ON pg.id = sp.parent_id 
                                    WHERE sp.student_id = ? AND sp.is_primary = 1");
        $stmt->execute([$student_id]);
        return $stmt->fetch();
    }

    // Parent User Accounts
    public function createParentUser($parent_id, $username, $password, $email) {
        $stmt = $this->db->prepare("INSERT INTO parent_users (parent_id, username, password, email) VALUES (?, ?, ?, ?)");
        return $stmt->execute([$parent_id, $username, password_hash($password, PASSWORD_BCRYPT), $email]);
    }

    public function getParentUser($username) {
        $stmt = $this->db->prepare("SELECT pu.*, pg.first_name, pg.last_name, pg.phone_primary 
                                    FROM parent_users pu 
                                    JOIN parents_guardians pg ON pu.parent_id = pg.id 
                                    WHERE pu.username = ?");
        $stmt->execute([$username]);
        return $stmt->fetch();
    }

    public function updateParentLastLogin($id) {
        $stmt = $this->db->prepare("UPDATE parent_users SET last_login = NOW() WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Statistics
    public function getTotalParents() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM parents_guardians WHERE status = 1");
        return $stmt->fetchColumn();
    }

    public function getParentCount() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM parents_guardians");
        return $stmt->fetchColumn();
    }
}