<?php
/**
 * Medical/Infirmary Model
 * Handles all medical-related database operations
 */
require_once __DIR__ . '/../../core/Database.php';

class MedicalModel {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Medical Staff
    public function getAllStaff() {
        $stmt = $this->db->query("SELECT * FROM medical_staff ORDER BY name ASC");
        return $stmt->fetchAll();
    }

    public function getStaffById($id) {
        $stmt = $this->db->prepare("SELECT * FROM medical_staff WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function createStaff($data) {
        $stmt = $this->db->prepare("INSERT INTO medical_staff (name, qualification, license_number, phone, email, specialization, status) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['name'], $data['qualification'], $data['license_number'], $data['phone'], $data['email'], $data['specialization'], $data['status']]);
    }

    public function updateStaff($id, $data) {
        $stmt = $this->db->prepare("UPDATE medical_staff SET name=?, qualification=?, license_number=?, phone=?, email=?, specialization=?, status=? WHERE id=?");
        return $stmt->execute([$data['name'], $data['qualification'], $data['license_number'], $data['phone'], $data['email'], $data['specialization'], $data['status'], $id]);
    }

    public function deleteStaff($id) {
        $stmt = $this->db->prepare("DELETE FROM medical_staff WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Medical Conditions
    public function getAllConditions() {
        $stmt = $this->db->query("SELECT * FROM medical_conditions ORDER BY condition_name ASC");
        return $stmt->fetchAll();
    }

    public function getConditionById($id) {
        $stmt = $this->db->prepare("SELECT * FROM medical_conditions WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function createCondition($data) {
        $stmt = $this->db->prepare("INSERT INTO medical_conditions (condition_name, condition_type, description, status) VALUES (?, ?, ?, ?)");
        return $stmt->execute([$data['condition_name'], $data['condition_type'], $data['description'], $data['status']]);
    }

    public function updateCondition($id, $data) {
        $stmt = $this->db->prepare("UPDATE medical_conditions SET condition_name=?, condition_type=?, description=?, status=? WHERE id=?");
        return $stmt->execute([$data['condition_name'], $data['condition_type'], $data['description'], $data['status'], $id]);
    }

    public function deleteCondition($id) {
        $stmt = $this->db->prepare("DELETE FROM medical_conditions WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Student Medical Records
    public function getMedicalRecord($student_id) {
        $stmt = $this->db->prepare("SELECT * FROM student_medical_records WHERE student_id = ?");
        $stmt->execute([$student_id]);
        return $stmt->fetch();
    }

    public function createMedicalRecord($data) {
        $stmt = $this->db->prepare("INSERT INTO student_medical_records (student_id, blood_group, height_cm, weight_kg, chronic_conditions, allergies, medications, emergency_contact_name, emergency_contact_phone, emergency_contact_relationship, family_medical_history, insurance_provider, insurance_policy_number, last_checkup_date, next_checkup_date, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['student_id'], $data['blood_group'], $data['height_cm'], $data['weight_kg'], $data['chronic_conditions'], $data['allergies'], $data['medications'], $data['emergency_contact_name'], $data['emergency_contact_phone'], $data['emergency_contact_relationship'], $data['family_medical_history'], $data['insurance_provider'], $data['insurance_policy_number'], $data['last_checkup_date'], $data['next_checkup_date'], $data['notes']]);
    }

    public function updateMedicalRecord($student_id, $data) {
        $stmt = $this->db->prepare("UPDATE student_medical_records SET blood_group=?, height_cm=?, weight_kg=?, chronic_conditions=?, allergies=?, medications=?, emergency_contact_name=?, emergency_contact_phone=?, emergency_contact_relationship=?, family_medical_history=?, insurance_provider=?, insurance_policy_number=?, last_checkup_date=?, next_checkup_date=?, notes=? WHERE student_id=?");
        return $stmt->execute([$data['blood_group'], $data['height_cm'], $data['weight_kg'], $data['chronic_conditions'], $data['allergies'], $data['medications'], $data['emergency_contact_name'], $data['emergency_contact_phone'], $data['emergency_contact_relationship'], $data['family_medical_history'], $data['insurance_provider'], $data['insurance_policy_number'], $data['last_checkup_date'], $data['next_checkup_date'], $data['notes'], $student_id]);
    }

    public function assignCondition($student_id, $condition_id, $diagnosed_date, $status = 'Active') {
        $stmt = $this->db->prepare("INSERT INTO student_medical_conditions (student_id, condition_id, diagnosed_date, status) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE status=?");
        return $stmt->execute([$student_id, $condition_id, $diagnosed_date, $status, $status]);
    }

    public function getStudentConditions($student_id) {
        $stmt = $this->db->prepare("SELECT smc.*, mc.condition_name, mc.condition_type FROM student_medical_conditions smc 
                                    JOIN medical_conditions mc ON smc.condition_id = mc.id 
                                    WHERE smc.student_id = ?");
        $stmt->execute([$student_id]);
        return $stmt->fetchAll();
    }

    // Medical Visits
    public function createVisit($data) {
        $stmt = $this->db->prepare("INSERT INTO medical_visits (student_id, visit_date, visit_time, complaint, diagnosis, treatment, medications_prescribed, referred_to, follow_up_date, status, attended_by, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['student_id'], $data['visit_date'], $data['visit_time'], $data['complaint'], $data['diagnosis'], $data['treatment'], $data['medications_prescribed'], $data['referred_to'], $data['follow_up_date'], $data['status'], $data['attended_by'], $data['notes']]);
    }

    public function getAllVisits($student_id = null, $status = null) {
        $sql = "SELECT mv.*, s.first_name, s.last_name, s.admission_no, ms.name as staff_name 
                FROM medical_visits mv 
                JOIN students s ON mv.student_id = s.id 
                LEFT JOIN medical_staff ms ON mv.attended_by = ms.id 
                WHERE 1=1";
        
        $params = [];
        if ($student_id) {
            $sql .= " AND mv.student_id = ?";
            $params[] = $student_id;
        }
        if ($status) {
            $sql .= " AND mv.status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY mv.visit_date DESC, mv.visit_time DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public function getVisitById($id) {
        $stmt = $this->db->prepare("SELECT mv.*, s.first_name, s.last_name, s.admission_no, ms.name as staff_name 
                                    FROM medical_visits mv 
                                    JOIN students s ON mv.student_id = s.id 
                                    LEFT JOIN medical_staff ms ON mv.attended_by = ms.id 
                                    WHERE mv.id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function updateVisit($id, $data) {
        $stmt = $this->db->prepare("UPDATE medical_visits SET diagnosis=?, treatment=?, medications_prescribed=?, referred_to=?, follow_up_date=?, status=?, notes=? WHERE id=?");
        return $stmt->execute([$data['diagnosis'], $data['treatment'], $data['medications_prescribed'], $data['referred_to'], $data['follow_up_date'], $data['status'], $data['notes'], $id]);
    }

    public function closeVisit($id) {
        $stmt = $this->db->prepare("UPDATE medical_visits SET status = 'Closed' WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Medications
    public function getAllMedications() {
        $stmt = $this->db->query("SELECT * FROM medications ORDER BY medication_name ASC");
        return $stmt->fetchAll();
    }

    public function getMedicationById($id) {
        $stmt = $this->db->prepare("SELECT * FROM medications WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function createMedication($data) {
        $stmt = $this->db->prepare("INSERT INTO medications (medication_name, generic_name, category, dosage_form, strength, stock_quantity, reorder_level, unit_price, supplier, expiry_date, batch_number, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['medication_name'], $data['generic_name'], $data['category'], $data['dosage_form'], $data['strength'], $data['stock_quantity'], $data['reorder_level'], $data['unit_price'], $data['supplier'], $data['expiry_date'], $data['batch_number'], $data['status']]);
    }

    public function updateMedication($id, $data) {
        $stmt = $this->db->prepare("UPDATE medications SET medication_name=?, generic_name=?, category=?, dosage_form=?, strength=?, stock_quantity=?, reorder_level=?, unit_price=?, supplier=?, expiry_date=?, batch_number=?, status=? WHERE id=?");
        return $stmt->execute([$data['medication_name'], $data['generic_name'], $data['category'], $data['dosage_form'], $data['strength'], $data['stock_quantity'], $data['reorder_level'], $data['unit_price'], $data['supplier'], $data['expiry_date'], $data['batch_number'], $data['status'], $id]);
    }

    public function updateStock($id, $quantity_change) {
        $stmt = $this->db->prepare("UPDATE medications SET stock_quantity = stock_quantity + ? WHERE id = ?");
        return $stmt->execute([$quantity_change, $id]);
    }

    public function deleteMedication($id) {
        $stmt = $this->db->prepare("DELETE FROM medications WHERE id = ?");
        return $stmt->execute([$id]);
    }

    public function getLowStockMedications() {
        $stmt = $this->db->query("SELECT * FROM medications WHERE stock_quantity <= reorder_level AND status = 1 ORDER BY stock_quantity ASC");
        return $stmt->fetchAll();
    }

    public function getExpiredMedications() {
        $stmt = $this->db->query("SELECT * FROM medications WHERE expiry_date < CURDATE() AND status = 1 ORDER BY expiry_date ASC");
        return $stmt->fetchAll();
    }

    // Medication Dispensing
    public function dispenseMedication($data) {
        $stmt = $this->db->prepare("INSERT INTO medication_dispensing (visit_id, medication_id, student_id, quantity, dosage_instructions, duration_days, dispensed_by) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['visit_id'], $data['medication_id'], $data['student_id'], $data['quantity'], $data['dosage_instructions'], $data['duration_days'], $data['dispensed_by']]);
    }

    public function getDispensingHistory($student_id = null) {
        $sql = "SELECT md.*, m.medication_name, mv.visit_date, s.first_name, s.last_name 
                FROM medication_dispensing md 
                JOIN medications m ON md.medication_id = m.id 
                JOIN medical_visits mv ON md.visit_id = mv.id 
                JOIN students s ON md.student_id = s.id 
                WHERE 1=1";
        
        if ($student_id) {
            $sql .= " AND md.student_id = ?";
        }
        
        $sql .= " ORDER BY md.dispensed_date DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($student_id ? [$student_id] : []);
        return $stmt->fetchAll();
    }

    // Inventory Log
    public function logInventory($data) {
        $stmt = $this->db->prepare("INSERT INTO medication_inventory_log (medication_id, transaction_type, quantity, balance_after, reference_number, notes, created_by) VALUES (?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['medication_id'], $data['transaction_type'], $data['quantity'], $data['balance_after'], $data['reference_number'], $data['notes'], $data['created_by']]);
    }

    public function getInventoryLog($medication_id = null) {
        $sql = "SELECT mil.*, m.medication_name, u.full_name as created_by_name 
                FROM medication_inventory_log mil 
                JOIN medications m ON mil.medication_id = m.id 
                JOIN users u ON mil.created_by = u.id 
                WHERE 1=1";
        
        if ($medication_id) {
            $sql .= " AND mil.medication_id = ?";
        }
        
        $sql .= " ORDER BY mil.created_at DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($medication_id ? [$medication_id] : []);
        return $stmt->fetchAll();
    }

    // Immunization
    public function createImmunization($data) {
        $stmt = $this->db->prepare("INSERT INTO immunization_records (student_id, vaccine_name, dose_number, administration_date, next_due_date, administered_by, location, batch_number, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['student_id'], $data['vaccine_name'], $data['dose_number'], $data['administration_date'], $data['next_due_date'], $data['administered_by'], $data['location'], $data['batch_number'], $data['notes']]);
    }

    public function getImmunizations($student_id) {
        $stmt = $this->db->prepare("SELECT * FROM immunization_records WHERE student_id = ? ORDER BY administration_date DESC");
        $stmt->execute([$student_id]);
        return $stmt->fetchAll();
    }

    public function getDueImmunizations() {
        $stmt = $this->db->query("SELECT ir.*, s.first_name, s.last_name, s.admission_no 
                                  FROM immunization_records ir 
                                  JOIN students s ON ir.student_id = s.id 
                                  WHERE ir.next_due_date <= CURDATE() 
                                  ORDER BY ir.next_due_date ASC");
        return $stmt->fetchAll();
    }

    // Medical Incidents
    public function createIncident($data) {
        $stmt = $this->db->prepare("INSERT INTO medical_incidents (student_id, incident_date, incident_time, incident_type, description, location, witnesses, action_taken, reported_by, parent_notified, hospital_referral, hospital_name, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$data['student_id'], $data['incident_date'], $data['incident_time'], $data['incident_type'], $data['description'], $data['location'], $data['witnesses'], $data['action_taken'], $data['reported_by'], $data['parent_notified'], $data['hospital_referral'], $data['hospital_name'], $data['status']]);
    }

    public function getAllIncidents($status = null) {
        $sql = "SELECT mi.*, s.first_name, s.last_name, s.admission_no, u.full_name as reported_by_name 
                FROM medical_incidents mi 
                JOIN students s ON mi.student_id = s.id 
                JOIN users u ON mi.reported_by = u.id 
                WHERE 1=1";
        
        if ($status) {
            $sql .= " AND mi.status = ?";
        }
        
        $sql .= " ORDER BY mi.incident_date DESC, mi.incident_time DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($status ? [$status] : []);
        return $stmt->fetchAll();
    }

    public function getIncidentById($id) {
        $stmt = $this->db->prepare("SELECT mi.*, s.first_name, s.last_name, s.admission_no, u.full_name as reported_by_name 
                                    FROM medical_incidents mi 
                                    JOIN students s ON mi.student_id = s.id 
                                    JOIN users u ON mi.reported_by = u.id 
                                    WHERE mi.id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function updateIncident($id, $data) {
        $stmt = $this->db->prepare("UPDATE medical_incidents SET action_taken=?, parent_notified=?, parent_notification_date=?, hospital_referral=?, hospital_name=?, status=? WHERE id=?");
        return $stmt->execute([$data['action_taken'], $data['parent_notified'], $data['parent_notification_date'], $data['hospital_referral'], $data['hospital_name'], $data['status'], $id]);
    }

    public function closeIncident($id) {
        $stmt = $this->db->prepare("UPDATE medical_incidents SET status = 'Closed' WHERE id = ?");
        return $stmt->execute([$id]);
    }

    // Statistics
    public function getTotalVisits() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM medical_visits");
        return $stmt->fetchColumn();
    }

    public function getTotalStudentsWithRecords() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM student_medical_records");
        return $stmt->fetchColumn();
    }

    public function getVisitsThisMonth() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM medical_visits WHERE MONTH(visit_date) = MONTH(CURDATE()) AND YEAR(visit_date) = YEAR(CURDATE())");
        return $stmt->fetchColumn();
    }

    public function getActiveConditions() {
        $stmt = $this->db->query("SELECT COUNT(*) FROM student_medical_conditions WHERE status = 'Active'");
        return $stmt->fetchColumn();
    }

    public function getMedicationValue() {
        $stmt = $this->db->query("SELECT SUM(stock_quantity * unit_price) FROM medications WHERE status = 1");
        return $stmt->fetchColumn() ?? 0;
    }
}