<?php
/**
 * File Upload Helper
 * Handles secure file uploads
 */
class FileHelper {
    /**
     * Upload a file securely
     */
    public static function upload($file, $destination, $allowed_types = ['jpg', 'jpeg', 'png', 'pdf']) {
        if (!isset($file) || $file['error'] != UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'File upload failed'];
        }
        
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, $allowed_types)) {
            return ['success' => false, 'message' => 'Invalid file type. Allowed: ' . implode(', ', $allowed_types)];
        }
        
        if ($file['size'] > UPLOAD_MAX_SIZE) {
            return ['success' => false, 'message' => 'File too large. Max size: ' . (UPLOAD_MAX_SIZE / 1024 / 1024) . 'MB'];
        }
        
        // Generate unique filename
        $new_filename = uniqid() . '_' . time() . '.' . $file_extension;
        $upload_path = $destination . '/' . $new_filename;
        
        // Create directory if not exists
        if (!is_dir($destination)) {
            mkdir($destination, 0755, true);
        }
        
        if (move_uploaded_file($file['tmp_name'], $upload_path)) {
            return ['success' => true, 'filename' => $new_filename, 'path' => $upload_path];
        }
        
        return ['success' => false, 'message' => 'Failed to save file'];
    }
    
    /**
     * Delete a file
     */
    public static function delete($filepath) {
        if (file_exists($filepath)) {
            return unlink($filepath);
        }
        return false;
    }
    
    /**
     * Get file size in human readable format
     */
    public static function fileSize($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, 2) . ' ' . $units[$pow];
    }
}