<?php
/**
 * Transport Controller
 * Handles all transport-related operations
 */
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../core/Controller.php';
require_once __DIR__ . '/../../app/models/TransportModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/helpers/SessionHelper.php';

class TransportController extends Controller {
    private $model;
    private $studentModel;

    public function __construct() {
        Auth::checkRole(['Super Admin', 'Admin', 'Accountant']);
        $this->model = new TransportModel();
        $this->studentModel = new StudentModel();
    }

    public function index() {
        $vehicles = $this->model->getAllVehicles();
        require_once __DIR__ . '/../views/transport/vehicles.php';
    }

    public function routes() {
        $routes = $this->model->getAllRoutes();
        require_once __DIR__ . '/../views/transport/routes.php';
    }

    public function students() {
        $transports = $this->model->getAllStudentTransport();
        require_once __DIR__ . '/../views/transport/students.php';
    }

    // Vehicle Methods
    public function createVehicle() {
        require_once __DIR__ . '/../views/transport/vehicle_create.php';
    }

    public function storeVehicle() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createVehicle($_POST)) {
                SessionHelper::setFlash('transport_msg', 'Vehicle added successfully!', 'success');
            } else {
                SessionHelper::setFlash('transport_msg', 'Failed to add vehicle.', 'danger');
            }
            header("Location: /sms/public/transport/index");
            exit;
        }
    }

    public function editVehicle($id) {
        $vehicle = $this->model->getVehicleById($id);
        if (!$vehicle) die("Vehicle not found");
        require_once __DIR__ . '/../views/transport/vehicle_edit.php';
    }

    public function updateVehicle($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateVehicle($id, $_POST)) {
                SessionHelper::setFlash('transport_msg', 'Vehicle updated successfully!', 'success');
            }
            header("Location: /sms/public/transport/index");
            exit;
        }
    }

    public function deleteVehicle($id) {
        if ($this->model->deleteVehicle($id)) {
            SessionHelper::setFlash('transport_msg', 'Vehicle deleted successfully!', 'success');
        }
        header("Location: /sms/public/transport/index");
        exit;
    }

    // Route Methods
    public function createRoute() {
        require_once __DIR__ . '/../views/transport/route_create.php';
    }

    public function storeRoute() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createRoute($_POST)) {
                SessionHelper::setFlash('transport_msg', 'Route added successfully!', 'success');
            }
            header("Location: /sms/public/transport/routes");
            exit;
        }
    }

    public function editRoute($id) {
        $route = $this->model->getRouteById($id);
        if (!$route) die("Route not found");
        require_once __DIR__ . '/../views/transport/route_edit.php';
    }

    public function updateRoute($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateRoute($id, $_POST)) {
                SessionHelper::setFlash('transport_msg', 'Route updated successfully!', 'success');
            }
            header("Location: /sms/public/transport/routes");
            exit;
        }
    }

    public function deleteRoute($id) {
        if ($this->model->deleteRoute($id)) {
            SessionHelper::setFlash('transport_msg', 'Route deleted successfully!', 'success');
        }
        header("Location: /sms/public/transport/routes");
        exit;
    }

    // Student Transport Assignment
    public function assign() {
        $students = $this->studentModel->getAll();
        $vehicles = $this->model->getAllVehicles();
        $routes = $this->model->getAllRoutes();
        require_once __DIR__ . '/../views/transport/assign.php';
    }

    public function storeAssignment() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            
            // Check vehicle capacity
            $occupancy = $this->model->getVehicleOccupancy($_POST['vehicle_id']);
            $capacity = $this->model->getVehicleCapacity($_POST['vehicle_id']);
            
            if ($occupancy >= $capacity) {
                SessionHelper::setFlash('transport_msg', 'Vehicle is at full capacity!', 'danger');
                header("Location: /sms/public/transport/assign");
                exit;
            }
            
            if ($this->model->assignTransport($_POST)) {
                SessionHelper::setFlash('transport_msg', 'Student assigned to transport successfully!', 'success');
            } else {
                SessionHelper::setFlash('transport_msg', 'Failed to assign student.', 'danger');
            }
            header("Location: /sms/public/transport/students");
            exit;
        }
    }

    public function removeAssignment($id) {
        if ($this->model->updateTransportStatus($id, 0)) {
            SessionHelper::setFlash('transport_msg', 'Transport assignment removed!', 'success');
        }
        header("Location: /sms/public/transport/students");
        exit;
    }
}