<?php
/**
 * Result Controller
 * Handles exam results and report cards
 */
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../core/Controller.php';
require_once __DIR__ . '/../../core/Database.php';
require_once __DIR__ . '/../../app/models/ResultModel.php';
require_once __DIR__ . '/../../app/models/ExamModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/helpers/SessionHelper.php';

class ResultController extends Controller {
    private $resultModel;
    private $examModel;
    private $studentModel;
    private $db;

    public function __construct() {
        $this->resultModel = new ResultModel();
        $this->examModel = new ExamModel();
        $this->studentModel = new StudentModel();
        $this->db = Database::getInstance()->getConnection();
    }

    public function index() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher', 'Student', 'Parent']);
        require_once __DIR__ . '/../views/results/index.php';
    }

    // Teachers enter marks
    public function enter() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        $exams = $this->examModel->getAll();
        $classes = $this->studentModel->getClasses();
        $subjects = $this->examModel->getSubjectsByClass(1); // Default, will be loaded via AJAX
        require_once __DIR__ . '/../views/results/enter.php';
    }

    public function loadMarks() {
        // AJAX Endpoint to load students for marking
        header('Content-Type: application/json');
        $exam_id = $_GET['exam_id'] ?? 0;
        $class_id = $_GET['class_id'] ?? 0;
        $subject_id = $_GET['subject_id'] ?? 0;

        // Get students in this class
        $stmt = $this->db->prepare("SELECT id, first_name, last_name, admission_no FROM students WHERE class_id = ?");
        $stmt->execute([$class_id]);
        $students = $stmt->fetchAll();

        // Get existing marks
        $stmt = $this->db->prepare("SELECT student_id, marks_obtained FROM exam_results WHERE exam_id = ? AND subject_id = ?");
        $stmt->execute([$exam_id, $subject_id]);
        $marks = $stmt->fetchAll(PDO::FETCH_KEY_PAIR); // [student_id => marks]

        echo json_encode(['students' => $students, 'marks' => $marks]);
        exit;
    }

    public function save() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            
            $exam_id = $_POST['exam_id'];
            $subject_id = $_POST['subject_id'];
            $marks = $_POST['marks']; // Array [student_id => marks]
            $user = Auth::user();

            foreach ($marks as $student_id => $score) {
                if (!empty($score) && is_numeric($score)) {
                    // Calculate grade
                    $grade = $this->resultModel->getGrade($score);
                    
                    $this->resultModel->saveMarks($exam_id, $student_id, $subject_id, $score, $grade['grade_name'] ?? '', $grade['point'] ?? 0, $user['id']);
                }
            }

            SessionHelper::setFlash('result_msg', 'Marks saved successfully!', 'success');
            header("Location: /sms/public/result/enter");
            exit;
        }
    }

    // Students/Parents view report
    public function report() {
        $user = Auth::user();
        $student_id = $_GET['id'] ?? null;

        // If no student ID provided, show search form
        if (!$student_id) {
            // For students, try to get their linked student record
            if ($user['role_name'] == 'Student') {
                // In production, fetch from user-student relationship table
                // For demo, require student ID in URL
            }
            require_once __DIR__ . '/../views/results/search.php';
            return;
        }

        $results = $this->resultModel->getStudentResults($student_id);
        $student = $this->studentModel->getById($student_id);
        
        if (!$student) {
            SessionHelper::setFlash('result_msg', 'Student not found.', 'danger');
            header("Location: /sms/public/result/report");
            exit;
        }
        
        require_once __DIR__ . '/../views/results/report.php';
    }

    public function search() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        $exams = $this->examModel->getAll();
        $classes = $this->studentModel->getClasses();
        require_once __DIR__ . '/../views/results/search.php';
    }

    public function searchResults() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        
        if ($_SERVER['REQUEST_METHOD'] == 'GET') {
            $exam_id = $_GET['exam_id'] ?? null;
            $class_id = $_GET['class_id'] ?? null;
            $student_id = $_GET['student_id'] ?? null;

            $sql = "SELECT er.*, e.name as exam_name, s.first_name, s.last_name, s.admission_no, sub.name as subject_name
                    FROM exam_results er
                    JOIN exams e ON er.exam_id = e.id
                    JOIN students s ON er.student_id = s.id
                    JOIN subjects sub ON er.subject_id = sub.id
                    WHERE 1=1";
            
            $params = [];
            
            if ($exam_id) {
                $sql .= " AND er.exam_id = ?";
                $params[] = $exam_id;
            }
            if ($class_id) {
                $sql .= " AND s.class_id = ?";
                $params[] = $class_id;
            }
            if ($student_id) {
                $sql .= " AND er.student_id = ?";
                $params[] = $student_id;
            }
            
            $sql .= " ORDER BY e.year DESC, e.term DESC, s.admission_no ASC";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            $results = $stmt->fetchAll();
            
            $exams = $this->examModel->getAll();
            $classes = $this->studentModel->getClasses();
            
            require_once __DIR__ . '/../views/results/search_results.php';
        }
    }

    public function edit($id) {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        
        $stmt = $this->db->prepare("SELECT er.*, e.name as exam_name, s.first_name, s.last_name, sub.name as subject_name
                                    FROM exam_results er
                                    JOIN exams e ON er.exam_id = e.id
                                    JOIN students s ON er.student_id = s.id
                                    JOIN subjects sub ON er.subject_id = sub.id
                                    WHERE er.id = ?");
        $stmt->execute([$id]);
        $result = $stmt->fetch();
        
        if (!$result) {
            SessionHelper::setFlash('result_msg', 'Result not found.', 'danger');
            header("Location: /sms/public/result/index");
            exit;
        }
        
        require_once __DIR__ . '/../views/results/edit.php';
    }

    public function update($id) {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            
            $marks = $_POST['marks_obtained'];
            $grade = $this->resultModel->getGrade($marks);
            
            $stmt = $this->db->prepare("UPDATE exam_results SET marks_obtained = ?, grade = ?, points = ?, entered_by = ? WHERE id = ?");
            $stmt->execute([$marks, $grade['grade_name'] ?? '', $grade['point'] ?? 0, Auth::user()['id'], $id]);
            
            SessionHelper::setFlash('result_msg', 'Result updated successfully!', 'success');
            header("Location: /sms/public/result/index");
            exit;
        }
    }

    public function delete($id) {
        Auth::checkRole(['Super Admin', 'Admin']);
        
        $stmt = $this->db->prepare("DELETE FROM exam_results WHERE id = ?");
        if ($stmt->execute([$id])) {
            SessionHelper::setFlash('result_msg', 'Result deleted successfully!', 'success');
        } else {
            SessionHelper::setFlash('result_msg', 'Failed to delete result.', 'danger');
        }
        header("Location: /sms/public/result/index");
        exit;
    }

    public function bulkEntry() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        $exams = $this->examModel->getAll();
        $classes = $this->studentModel->getClasses();
        require_once __DIR__ . '/../views/results/bulk_entry.php';
    }

    public function printReport($student_id) {
        $student = $this->studentModel->getById($student_id);
        $results = $this->resultModel->getStudentResults($student_id);
        
        if (!$student) {
            die("Student not found");
        }
        
        require_once __DIR__ . '/../views/results/print_report.php';
    }
}