<?php
/**
 * Parent/Guardian Controller
 * Handles all parent-related operations
 */
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../core/Controller.php';
require_once __DIR__ . '/../../app/models/ParentModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/helpers/SessionHelper.php';

class ParentController extends Controller {
    private $model;
    private $studentModel;

    public function __construct() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher', 'Accountant']);
        $this->model = new ParentModel();
        $this->studentModel = new StudentModel();
    }

    public function index() {
        $parents = $this->model->getAll();
        require_once __DIR__ . '/../views/parents/index.php';
    }

    public function search() {
        $keyword = $_GET['q'] ?? '';
        $parents = $keyword ? $this->model->search($keyword) : $this->model->getAll();
        require_once __DIR__ . '/../views/parents/index.php';
    }

    public function create() {
        require_once __DIR__ . '/../views/parents/create.php';
    }

    public function store() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            
            if ($this->model->create($_POST)) {
                SessionHelper::setFlash('parent_msg', 'Parent/Guardian added successfully!', 'success');
            } else {
                SessionHelper::setFlash('parent_msg', 'Failed to add parent/guardian.', 'danger');
            }
            header("Location: /sms/public/parent/index");
            exit;
        }
    }

    public function edit($id) {
        $parent = $this->model->getById($id);
        if (!$parent) die("Parent not found");
        require_once __DIR__ . '/../views/parents/edit.php';
    }

    public function update($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->update($id, $_POST)) {
                SessionHelper::setFlash('parent_msg', 'Parent/Guardian updated successfully!', 'success');
            }
            header("Location: /sms/public/parent/index");
            exit;
        }
    }

    public function delete($id) {
        if ($this->model->delete($id)) {
            SessionHelper::setFlash('parent_msg', 'Parent/Guardian deleted successfully!', 'success');
        }
        header("Location: /sms/public/parent/index");
        exit;
    }

    public function view($id) {
        $parent = $this->model->getById($id);
        $students = $this->model->getStudentsByParent($id);
        if (!$parent) die("Parent not found");
        require_once __DIR__ . '/../views/parents/view.php';
    }

    public function assign() {
        $parents = $this->model->getAll();
        $students = $this->studentModel->getAll();
        require_once __DIR__ . '/../views/parents/assign.php';
    }

    public function storeAssignment() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->assignParent($_POST['student_id'], $_POST['parent_id'], $_POST['is_primary'] ?? 0)) {
                SessionHelper::setFlash('parent_msg', 'Parent assigned to student successfully!', 'success');
            }
            header("Location: /sms/public/parent/assignments");
            exit;
        }
    }

    public function assignments() {
        $db = Database::getInstance()->getConnection();
        $assignments = $db->query("SELECT sp.*, pg.first_name, pg.last_name, pg.phone_primary, s.first_name as student_first, s.last_name as student_last, s.admission_no 
                                   FROM student_parents sp 
                                   JOIN parents_guardians pg ON sp.parent_id = pg.id 
                                   JOIN students s ON sp.student_id = s.id 
                                   ORDER BY sp.created_at DESC")->fetchAll();
        require_once __DIR__ . '/../views/parents/assignments.php';
    }

    public function removeAssignment($student_id, $parent_id) {
        if ($this->model->removeParent($student_id, $parent_id)) {
            SessionHelper::setFlash('parent_msg', 'Assignment removed successfully!', 'success');
        }
        header("Location: /sms/public/parent/assignments");
        exit;
    }

    public function createAccount() {
        $parents = $this->model->getAll();
        require_once __DIR__ . '/../views/parents/create_account.php';
    }

    public function storeAccount() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createParentUser($_POST['parent_id'], $_POST['username'], $_POST['password'], $_POST['email'])) {
                SessionHelper::setFlash('parent_msg', 'Parent account created successfully!', 'success');
            } else {
                SessionHelper::setFlash('parent_msg', 'Failed to create account. Username or email may already exist.', 'danger');
            }
            header("Location: /sms/public/parent/index");
            exit;
        }
    }
}