<?php
/**
 * Library Controller
 * Handles all library-related operations
 */
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../core/Controller.php';
require_once __DIR__ . '/../../app/models/LibraryModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/helpers/SessionHelper.php';

class LibraryController extends Controller {
    private $model;
    private $studentModel;

    public function __construct() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher', 'Student']);
        $this->model = new LibraryModel();
        $this->studentModel = new StudentModel();
    }

    public function index() {
        $books = $this->model->getAllBooks();
        require_once __DIR__ . '/../views/library/books.php';
    }

    public function search() {
        $keyword = $_GET['q'] ?? '';
        $books = $keyword ? $this->model->searchBooks($keyword) : $this->model->getAllBooks();
        require_once __DIR__ . '/../views/library/books.php';
    }

    public function issues() {
        $status = $_GET['status'] ?? null;
        $issues = $this->model->getBookIssues(null, null, $status);
        require_once __DIR__ . '/../views/library/issues.php';
    }

    public function overdue() {
        $overdue = $this->model->getOverdueBooks();
        require_once __DIR__ . '/../views/library/overdue.php';
    }

    // Book Methods
    public function createBook() {
        require_once __DIR__ . '/../views/library/book_create.php';
    }

    public function storeBook() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createBook($_POST)) {
                SessionHelper::setFlash('library_msg', 'Book added successfully!', 'success');
            } else {
                SessionHelper::setFlash('library_msg', 'Failed to add book.', 'danger');
            }
            header("Location: /sms/public/library/index");
            exit;
        }
    }

    public function editBook($id) {
        $book = $this->model->getBookById($id);
        if (!$book) die("Book not found");
        require_once __DIR__ . '/../views/library/book_edit.php';
    }

    public function updateBook($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateBook($id, $_POST)) {
                SessionHelper::setFlash('library_msg', 'Book updated successfully!', 'success');
            }
            header("Location: /sms/public/library/index");
            exit;
        }
    }

    public function deleteBook($id) {
        if ($this->model->deleteBook($id)) {
            SessionHelper::setFlash('library_msg', 'Book deleted successfully!', 'success');
        }
        header("Location: /sms/public/library/index");
        exit;
    }

    // Issue Methods
    public function issue() {
        $books = $this->model->getAllBooks();
        $students = $this->studentModel->getAll();
        require_once __DIR__ . '/../views/library/issue_book.php';
    }

    public function storeIssue() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            
            // Check if book is available
            $book = $this->model->getBookById($_POST['book_id']);
            if ($book['available_copies'] <= 0) {
                SessionHelper::setFlash('library_msg', 'No copies available!', 'danger');
                header("Location: /sms/public/library/issue");
                exit;
            }
            
            // Check student issue limit (max 3 books)
            $issueCount = $this->model->getStudentIssueCount($_POST['student_id']);
            if ($issueCount >= 3) {
                SessionHelper::setFlash('library_msg', 'Student has reached maximum issue limit (3 books)', 'danger');
                header("Location: /sms/public/library/issue");
                exit;
            }
            
            if ($this->model->issueBook($_POST)) {
                $this->model->updateAvailability($_POST['book_id'], -1);
                SessionHelper::setFlash('library_msg', 'Book issued successfully!', 'success');
            } else {
                SessionHelper::setFlash('library_msg', 'Failed to issue book.', 'danger');
            }
            header("Location: /sms/public/library/issues");
            exit;
        }
    }

    public function returnBook($id) {
        $issue = $this->model->getIssueById($id);
        require_once __DIR__ . '/../views/library/return_book.php';
    }

    public function processReturn($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            
            $issue = $this->model->getIssueById($id);
            $return_date = $_POST['return_date'];
            $fine = $this->model->calculateFine($issue['due_date'], $return_date);
            
            if ($this->model->returnBook($id, $return_date, $fine)) {
                $this->model->updateAvailability($issue['book_id'], 1);
                SessionHelper::setFlash('library_msg', 'Book returned successfully!' . ($fine > 0 ? " Fine: $" . $fine : ''), 'success');
            } else {
                SessionHelper::setFlash('library_msg', 'Failed to process return.', 'danger');
            }
            header("Location: /sms/public/library/issues");
            exit;
        }
    }
}