<?php
/**
 * Boarding/Hostel Controller
 * Handles all boarding-related operations
 */
require_once __DIR__ . '/../../core/Auth.php';
require_once __DIR__ . '/../../core/Controller.php';
require_once __DIR__ . '/../../app/models/BoardingModel.php';
require_once __DIR__ . '/../../app/models/StudentModel.php';
require_once __DIR__ . '/../../app/helpers/SessionHelper.php';

class BoardingController extends Controller {
    private $model;
    private $studentModel;

    public function __construct() {
        Auth::checkRole(['Super Admin', 'Admin', 'Teacher']);
        $this->model = new BoardingModel();
        $this->studentModel = new StudentModel();
    }

    public function index() {
        $hostels = $this->model->getAllHostels();
        require_once __DIR__ . '/../views/boarding/hostels.php';
    }

    public function rooms() {
        $hostel_id = $_GET['hostel'] ?? null;
        $rooms = $this->model->getAllRooms($hostel_id);
        $hostels = $this->model->getAllHostels();
        require_once __DIR__ . '/../views/boarding/rooms.php';
    }

    public function students() {
        $students = $this->model->getBoardingStudents();
        require_once __DIR__ . '/../views/boarding/students.php';
    }

    public function inspections() {
        $db = Database::getInstance()->getConnection();
        $inspections = $db->query("SELECT ri.*, r.room_number, h.name as hostel_name, u.full_name as inspector_name 
                                   FROM room_inspections ri 
                                   JOIN rooms r ON ri.room_id = r.id 
                                   JOIN hostels h ON r.hostel_id = h.id 
                                   JOIN users u ON ri.inspector_id = u.id 
                                   ORDER BY ri.inspection_date DESC")->fetchAll();
        require_once __DIR__ . '/../views/boarding/inspections.php';
    }

    // Hostel Methods
    public function createHostel() {
        require_once __DIR__ . '/../views/boarding/hostel_create.php';
    }

    public function storeHostel() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createHostel($_POST)) {
                SessionHelper::setFlash('boarding_msg', 'Hostel added successfully!', 'success');
            }
            header("Location: /sms/public/boarding/index");
            exit;
        }
    }

    public function editHostel($id) {
        $hostel = $this->model->getHostelById($id);
        if (!$hostel) die("Hostel not found");
        require_once __DIR__ . '/../views/boarding/hostel_edit.php';
    }

    public function updateHostel($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateHostel($id, $_POST)) {
                SessionHelper::setFlash('boarding_msg', 'Hostel updated successfully!', 'success');
            }
            header("Location: /sms/public/boarding/index");
            exit;
        }
    }

    public function deleteHostel($id) {
        if ($this->model->deleteHostel($id)) {
            SessionHelper::setFlash('boarding_msg', 'Hostel deleted successfully!', 'success');
        }
        header("Location: /sms/public/boarding/index");
        exit;
    }

    // Room Methods
    public function createRoom() {
        $hostels = $this->model->getAllHostels();
        require_once __DIR__ . '/../views/boarding/room_create.php';
    }

    public function storeRoom() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->createRoom($_POST)) {
                SessionHelper::setFlash('boarding_msg', 'Room added successfully!', 'success');
            }
            header("Location: /sms/public/boarding/rooms");
            exit;
        }
    }

    public function editRoom($id) {
        $room = $this->model->getRoomById($id);
        $hostels = $this->model->getAllHostels();
        if (!$room) die("Room not found");
        require_once __DIR__ . '/../views/boarding/room_edit.php';
    }

    public function updateRoom($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            if ($this->model->updateRoom($id, $_POST)) {
                SessionHelper::setFlash('boarding_msg', 'Room updated successfully!', 'success');
            }
            header("Location: /sms/public/boarding/rooms");
            exit;
        }
    }

    public function deleteRoom($id) {
        if ($this->model->deleteRoom($id)) {
            SessionHelper::setFlash('boarding_msg', 'Room deleted successfully!', 'success');
        }
        header("Location: /sms/public/boarding/rooms");
        exit;
    }

    // Student Assignment
    public function assign() {
        $students = $this->studentModel->getAll();
        $rooms = $this->model->getAvailableRooms();
        require_once __DIR__ . '/../views/boarding/assign.php';
    }

    public function storeAssignment() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            
            // Check if student already has boarding
            $existing = $this->model->getStudentBoarding($_POST['student_id']);
            if ($existing) {
                SessionHelper::setFlash('boarding_msg', 'Student already has boarding assignment!', 'danger');
                header("Location: /sms/public/boarding/assign");
                exit;
            }
            
            // Check room capacity
            $room = $this->model->getRoomById($_POST['room_id']);
            if ($room['current_occupancy'] >= $room['capacity']) {
                SessionHelper::setFlash('boarding_msg', 'Room is at full capacity!', 'danger');
                header("Location: /sms/public/boarding/assign");
                exit;
            }
            
            if ($this->model->assignStudent($_POST)) {
                $this->model->updateOccupancy($_POST['room_id'], 1);
                SessionHelper::setFlash('boarding_msg', 'Student assigned to room successfully!', 'success');
            }
            header("Location: /sms/public/boarding/students");
            exit;
        }
    }

    public function checkout($id) {
        $boarding = $this->model->getBoardingById($id);
        require_once __DIR__ . '/../views/boarding/checkout.php';
    }

    public function processCheckout($id) {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $boarding = $this->model->getBoardingById($id);
            
            if ($this->model->checkoutStudent($id, $_POST['checkout_date'])) {
                $this->model->updateOccupancy($boarding['room_id'], -1);
                SessionHelper::setFlash('boarding_msg', 'Student checked out successfully!', 'success');
            }
            header("Location: /sms/public/boarding/students");
            exit;
        }
    }

    // Inspections
    public function createInspection() {
        $rooms = $this->model->getAllRooms();
        require_once __DIR__ . '/../views/boarding/inspection_create.php';
    }

    public function storeInspection() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            Auth::validateCsrf($_POST['csrf_token']);
            $user = Auth::user();
            
            $_POST['inspector_id'] = $user['id'];
            
            if ($this->model->createInspection($_POST)) {
                SessionHelper::setFlash('boarding_msg', 'Inspection recorded successfully!', 'success');
            }
            header("Location: /sms/public/boarding/inspections");
            exit;
        }
    }

    public function resolveInspection($id) {
        if ($this->model->updateInspectionStatus($id, 'Resolved')) {
            SessionHelper::setFlash('boarding_msg', 'Inspection marked as resolved!', 'success');
        }
        header("Location: /sms/public/boarding/inspections");
        exit;
    }
}